<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * This migration creates a trigger on the `rvw_article_flow` table.
     * The trigger runs AFTER a row is deleted.
     *
     * Purpose:
     *   - Automatically delete related records from the `rvw_article_flag` table
     *     where:
     *       - `aflg_aflw_id` matches the deleted row’s ID
     *       - `aflg_type` equals "g"
     *
     * Why:
     *   - Prevents orphaned rows in `rvw_article_flag`.
     *   - Keeps data consistent between article flow and flags.
     *
     * Notes:
     *   - We always drop the trigger first (IF EXISTS) to avoid errors
     *     when re-running migrations or refreshing the database.
     *   - MySQL ensures trigger names are unique within a schema,
     *     so checking by table is not necessary.
     */
    public function up(): void
    {
        DB::unprepared('DROP TRIGGER IF EXISTS trg_after_delete_rvw_article_flow');
        DB::unprepared('
            CREATE TRIGGER trg_after_delete_rvw_article_flow
            AFTER DELETE ON rvw_article_flow
            FOR EACH ROW
            BEGIN
                DELETE FROM rvw_article_flag
                WHERE aflg_aflw_id = OLD.aflw_id
                  AND aflg_type = "g";
            END
        ');
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
          DB::unprepared('DROP TRIGGER IF EXISTS trg_after_delete_rvw_article_flow');

    }
};
