<?php

namespace App\Http\Services;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Manage_Email_Api;
use App\Http\Traits\GeneralEmailTrait;
use App\Models\Articles;
use App\Models\Authors;
use App\Models\ConveyData;
use App\Models\Journals;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use FontLib\Table\Type\nameRecord;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;
use SimpleXMLElement;
use Sabre\Xml\Service as XMLService;

class ConveyService
{

    use GeneralEmailTrait;

    private $baseConveyUrl;
    private $webConveyUrl;
    private $policyId;
    private $discloserListId;
    private $authToken;

    public function __construct()
    {
        $this->baseConveyUrl = config('conveyServiceConfig.convey_base_url')[env('APP_ENV') == 'production' ? 'production' : 'staging'] ?? '';
        $this->webConveyUrl = config('conveyServiceConfig.convey_web_url')[env('APP_ENV') == 'production' ? 'production' : 'staging'] ?? '';
        $this->policyId = config('conveyServiceConfig.convey_policy_id')[env('APP_ENV') == 'production' ? 'production' : 'staging'] ?? '';
        $this->discloserListId = config('conveyServiceConfig.convey_discloser_list_id')[env('APP_ENV') == 'production' ? 'production' : 'staging'] ?? '';
        $this->authToken = Cache::get('convey_auth_token', function () {
            return $this->authenticate();
        });
    }

    public function __call($method, $parameters)
    {
        $method = str_replace('Blinded', '', $method);
        if (method_exists($this, $method)) {
            $result = call_user_func_array([$this, $method], $parameters);
            return $this->processResultForBlinding($result);
        }

        throw new \BadMethodCallException("Method {$method} does not exist.");
    }

    protected function processResultForBlinding($result)
    {
        traverseNestedArray($result, function ($key, $value, &$nested_array = null) {
            $blinding_keys = ['identifier', 'firstName', 'middleName', 'lastName', 'name', 'email', 'title', 'subscriberFirstName', 'subscriberMiddleName', 'subscriberLastName', 'subscriberEmail', 'subscriberTitle', 'subscriberId'];
            if (in_array($key, $blinding_keys)) {
                $nested_array[$key] = 'anonymous';
            }
        });
        return $result;
    }

    /**
     * Authenticate and retrieve token
     *
     * @return string
     */
    private function authenticate(): string
    {
        $username = config('conveyServiceConfig.convey_username') ?? null;
        $password = config('conveyServiceConfig.convey_password') ?? null;
        $auth_url = config('conveyServiceConfig.convey_authentication_url')[env('APP_ENV') == 'production' ? 'production' : 'staging'] ?? null;

        if (!$auth_url || !$username || !$password) {
            throw new Exception('Convey service is not configured in the system, please check the convey service configurations.');
        }
        $response = Http::withHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'X-OpenAM-Username' => $username,
            'X-OpenAM-Password' => $password,
        ])->post($auth_url);
        if ($response->successful()) {
            $newToken = $response->json('tokenId');
            Cache::put('convey_auth_token', $newToken);
            return $newToken;
        }
        throw new \Exception('Authentication failed: ' . $response->body());
    }

    /**
     * Send GET request to the Convey API
     *
     * @param string $endpoint
     * @param array $payload
     * @return array
     */
    public function get(string $endpoint, array $payload = [], $type = null, $use_web_url = false)
    {
        $response = retry(2, function () use ($endpoint, $payload, $use_web_url) {
            $url = $use_web_url ? $this->webConveyUrl : $this->baseConveyUrl;
            $response = Http::withHeaders([
                // 'Cache-Control' => 'no-cache',
                'iPlanetDirectoryPro' => $this->authToken,
            ])->get("{$url}{$endpoint}", $payload);
            if ($response->status() === 401) {
                $this->authToken = $this->authenticate();
                throw new \Exception('Unauthorized (401) - Retrying...');
            }
            return $response;
        }, 100);

        if ($response->successful()) {
            return $type ? $response->body() : $response->json();
        }

        throw new \Exception('GET request failed: ' . $response->body());
    }

    /**
     * Send POST request to the Convey API
     *
     * @param string $endpoint
     * @param array $payload
     * @return array
     * @throws \Exception
     */
    public function post(string $endpoint, array $payload): array
    {
        $response = retry(3, function () use ($endpoint, $payload) {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'iPlanetDirectoryPro' => $this->authToken,
            ])->post("{$this->baseConveyUrl}{$endpoint}", $payload);

            if ($response->status() === 401) {
                $this->authToken = $this->authenticate();
                throw new \Exception('Unauthorized (401) - Retrying...');
            }

            return $response;
        }, 100);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('POST request failed: ' . $response->body());
    }

    /**
     * Send PUT request to the Convey API
     *
     * @param string $endpoint
     * @param array $payload
     * @return array
     * @throws \Exception
     */
    public function put(string $endpoint, array $payload): array
    {
        $response = retry(3, function () use ($endpoint, $payload) {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'iPlanetDirectoryPro' => $this->authToken,
            ])->put("{$this->baseConveyUrl}{$endpoint}", $payload);

            if ($response->status() === 401) {
                $this->authToken = $this->authenticate();
                throw new \Exception('Unauthorized (401) - Retrying...');
            }

            return $response;
        }, 100);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('PUT request failed: ' . $response->body());
    }

    /**
     * Send DELETE request to the Convey API
     *
     * @param string $endpoint
     * @return array
     * @throws \Exception
     */
    public function delete(string $endpoint): array
    {
        $response = retry(3, function () use ($endpoint) {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'iPlanetDirectoryPro' => $this->authToken,
            ])->delete("{$this->baseConveyUrl}{$endpoint}");

            if ($response->status() === 401) {
                $this->authToken = $this->authenticate();
                throw new \Exception('Unauthorized (401) - Retrying...');
            }

            return $response;
        }, 100);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('DELETE request failed: ' . $response->body());
    }

    /**
     * Validate request parameters
     *
     * @param array $rules
     * @param array $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    private function validateParams(array $rules, array $data): \Illuminate\Contracts\Validation\Validator
    {
        return Validator::make($data, $rules);
    }

    /**
     * List Groups for a Policy
     *
     * @param array $payload
     * @return array
     */
    public function listGroups(array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/list-groups", $payload);
    }

    /**
     * Create Groups for a Policy
     *
     
     * @param array $payload name, vendorId(optional)
     * @return array
     */
    public function createGroups(array $payload = []): array
    {
        $validation = Validator::make($payload, [
            'name' => 'required|string|max:255'
        ]);

        if ($validation->fails()) {
            throw new Exception($validation->errors());
        }
        return $this->post("/policies/{$this->policyId}/list-groups", $payload);
    }

    /**
     * update group details
     *
     
     * @param int $groupId
     * @param array $payload name, vendorId(optional)
     * @return array
     */
    public function updateGroup(int $groupId, array $payload = []): array
    {
        return $this->put("/policies/{$this->policyId}/list-groups/{$groupId}", $payload);
    }


    /**
     * get group details
     *
     
     * @param int $groupId
     * @param array $payload
     * @return array
     */
    public function getGroups(int $groupId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/list-groups/{$groupId}", $payload);
    }

    /**
     * Paginated summary of Discloser Lists for a discloser list group.
     *
     
     * @param int $groupId
     * @param array $payload
     * @return array
     */
    public function listDiscloserListForListGroup(int $groupId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/list-groups/{$groupId}/discloser-lists", $payload);
    }

    /**
     * List Disclosers for a Policy
     *
     
     * @param array $payload
     * @return array
     */
    public function listDiscloserList(array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists", $payload);
    }

    /**
     * Create a Discloser list
     *
     * @param array $policyId 
     * @param array $payload name, vendorId(optional), purpose, contactName, contactEmail, submissionDeadline, startDate(optional), endDate(optional), status->id, status->description(optional), listGroupIds(optional), description(optional), interestsNotCollected, initialInviteTemplateId, dueDate, reminderTemplateId, reminderFrequency, reminderLimit, duplicationStatus
     * @return array
     */
    public function createDiscloserList(array $payload): array
    {
        return $this->post("/policies/{$this->policyId}/discloser-lists", $payload);
    }


    /**
     * get discloser list details
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function getDiscloserListDetail(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}", $payload);
    }


    /**
     * Update a Discloser List
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function updateDiscloserList(int $discloserListId, array $payload): array
    {
        return $this->put("/policies/{$this->policyId}/discloser-lists/{$discloserListId}", $payload);
    }

    /**
     * Get a discloser list activity report for a policy.
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function getDiscloserListActivityReport(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/activity-report", $payload);
    }

    /**
     * get discloser list details
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function duplicateDiscloserList(int $discloserListId, array $payload = []): array
    {
        return $this->post("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/duplicate", $payload);
    }

    /**
     * Get the bypass interests list text for a specific list.
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function getBypassInterestsListText(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/list-text", $payload);
    }

    /**
     * Update the bypass interests list text for a specific list.
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function updateBypassInterestsListText(int $discloserListId, array $payload = []): array
    {
        return $this->post("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/list-text", $payload);
    }

    /**
     * Create an access token for viewing a discloser list report.
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function getReportToken(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/reports/tokens", $payload);
    }

    /**
     * Find disclosers within a specific discloser list
     *
     
     * @param int $discloserListId
     * @param array $payload
     * @return array
     */
    public function listDiscloser(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers", $payload);
    }

    /**
     * Create disclosers within a specific discloser list
     *
     
     * @param int $discloserListId
     * @param array $payload identifier, firstName, lastName, title(optional), email, status->id, status->description(optional), role(optional), requestSent(optional)
     * @return array
     */
    public function createDiscloser(int $discloserListId, array $payload = []): array
    {
        return $this->post("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers", $payload);
    }

    /**
     * Find disclosers within a specific discloser list
     *
     
     * @param int $discloserListId
     * @param int $discloserId
     * @param array $payload
     * @return array
     */
    public function getDiscloserDetail(int $discloserListId, int $discloserId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}", $payload);
    }

    /**
     * Update disclosers within a specific discloser list
     *
     
     * @param int $discloserListId
     * @param int $discloserId
     * @param array $payload
     * @return array
     */
    public function updateDiscloserDetail(int $discloserListId, int $discloserId, array $payload = []): array
    {
        return $this->put("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}", $payload);
    }

    /**
     * Returns the url for a discloser to disclose
     *
     
     * @param int $discloserListId
     * @param int $discloserId
     * @param array $payload add redirectUrl if needed
     * @return string
     */
    public function getDiscloserLink(int $discloserListId, int $discloserId, array $payload = []): string
    {
        $data =  $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}/disclosureUrl", $payload);
        return $data['url'];
    }

    /**
     * Returns currently assigned disclosure statuses for a discloser
     *
     
     * @param int $discloserListId
     * @param int $discloserId
     * @param array $payload
     * @return array
     */
    public function getDiscloserStatus(int $discloserListId, int $discloserId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}/statuses", $payload);
    }

    /**
     * Updates currently assigned disclosure statuses for a discloser
     *
     
     * @param int $discloserListId
     * @param int $discloserId
     * @param array $payload title,sort,active,description(optional)
     * @return array
     */
    public function updateDiscloserStatus(int $discloserListId, int $discloserId, array $payload = []): array
    {
        return $this->post("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}/statuses", $payload);
    }

    /**
     * List of all disclosure statuses defined for this policy.
     *
     
     * @param array $payload
     * @return array
     */
    public function listDiscloserStatus(array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/disclosure-statuses", $payload);
    }

    /**
     *add custom statuses for policy.
     *
     
     * @param array $payload title,sort,active,description(optional)
     * @return array
     */
    public function addCustomDiscloserStatus(array $payload = []): array
    {
        return $this->post("/policies/{$this->policyId}/disclosure-statuses", $payload);
    }

    /**
     *update custom statuses for policy.
     *
     
     * @param array $payload
     * @return array
     */
    public function updateCustomDiscloserStatus(array $payload = []): array
    {
        return $this->post("/policies/{$this->policyId}/disclosure-statuses", $payload);
    }

    /**
     *Get a specific disclosure in XML v2 format based on MEDBIQ schema spec.
     *
     
     * @param int $disclosureId
     * @param array $payload
     * @return 
     */
    public function getDisclosureV2XML(int $disclosureId, array $payload = [])
    {
        return $this->get("/api/policies/{$this->policyId}/xml/v2/disclosures/{$disclosureId}/disclosure.xml", $payload, 'file', true);
    }

    /**
     *Get final Disclosure PDF Form with supporting documentation.
     *
     
     * @param int $disclosureId
     * @param array $payload
     * @return array
     */
    public function getFinalDisclosurePDFWithDocs(int $discloserListId, int $discloserId, int $disclosureId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}/disclosures/{$disclosureId}/disclosure-form", $payload);
    }

    /**
     *Get a specific disclosure in XML format based on MEDBIQ schema spec.
     *
     
     * @param int $discloserListId
     * @param int $discloserId
     * @param int $disclosureId
     * @param array $payload
     * @return string
     */
    public function getDisclosureXML(int $discloserListId, int $discloserId, int $disclosureId, array $payload = [])
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}/disclosures/{$disclosureId}/disclosure.xml", $payload, 'xml');
    }

    public function getDisclosureObject(int $disclosureId)
    {
        $xml_data = $this->getDisclosureV2XML($disclosureId);
        $data =  $this->conveyXMLParser($xml_data);
        return $this->FormatParsedXML($data);
    }

    /**
     *Get primary Disclosure PDF Form without supporting documentation
     *
     * @param int $discloserListId
     * @param int $discloserId
     * @param int $disclosureId
     * @param array $payload
     * @return array
     */
    public function getPrimaryDisclosurePDFWithoutDocs(int $discloserListId, int $discloserId, int $disclosureId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}/disclosures/{$disclosureId}/primary-form", $payload);
    }

    /**
     *Get primary Disclosure PDF Form without supporting documentation
     *
     
     * @param int $discloserListId
     * @param int $discloserId
     * @param int $disclosureId
     * @param int $docId
     * @param array $payload
     * @return array
     */
    public function getSupportingDocsForDisclosure(int $discloserListId, int $discloserId, int $disclosureId, int $docId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers/{$discloserId}/disclosures/{$disclosureId}/supporting-documents/{$docId}", $payload);
    }

    /**
     * Retrieves all created email templates for a policy.
     *
     * @param array $payload
     * @return array
     */
    public function getEmailTemplates(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-list/{$discloserListId}/reports/tokens", $payload);
    }

    /**
     * Create an email template for a policy.
     *
     * @param array $payload description,messageSubject,messageBody,replyTo(optional)
     * @return array
     */
    public function createEmailTemplates(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-list/{$discloserListId}/reports/tokens", $payload);
    }

    /**
     * Update an email template for a policy.
     *
     * @param array $payload description,messageSubject,messageBody,replyTo(optional)
     * @return array
     */
    public function updateEmailTemplates(int $discloserListId, array $payload = []): array
    {
        return $this->get("/policies/{$this->policyId}/discloser-list/{$discloserListId}/reports/tokens", $payload);
    }

    public function getDiscloserByIdentifier(int $discloserListId, string $identifier)
    {

        $data = $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers", ['search' => "subscriberId==$identifier"],  null, true);
        return $data['content'][0] ?? null;
    }

    public function getDisclosersFromList(int $discloserListId)
    {
        $data = $this->get("/policies/{$this->policyId}/discloser-lists/{$discloserListId}/disclosers", [],  null, true);
        return $data['content'] ?? null;
    }

    public function getDisclosureById($disclosureId)
    {
        return $this->get("/policies/{$this->policyId}/disclosures/{$disclosureId}", use_web_url: true);
    }

    public function getAllDisclosureListApi($params){
        return $this->get("/policies/2569/disclosures",$params, use_web_url:true);
    }

    public function addUserToDiscloser(int $user_id, int $art_id, array $role = null)
    {
        $convey_user = null;
        $existing_discloser = ConveyData::where(['rcd_usr_id' => $user_id, 'rcd_art_id' => $art_id])->get(['rcd_discloser_id', 'rcd_disclosure_cmpltn_status'])?->first();
        $existing_list = ConveyData::where(['rcd_art_id' => $art_id])->get(['rcd_discloser_list_id'])?->first();
        if ($existing_list) {
            $discloser_list = $this->getDiscloserListDetail($existing_list->rcd_discloser_list_id);
        } else {
            $discloser_list = $this->createDiscloserListForArticle($art_id);
        }
        $discloserListId = $discloser_list['id'];
        if (!$existing_discloser) {
            $user = User::where('usr_id', $user_id)->get(['usr_title as title', 'usr_first_name as firstName', 'usr_middle_name as middleName', 'usr_last_name as lastName', 'usr_email as email', 'usr_email as identifier', 'usr_status as status', 'usr_orcid_id as orcid_id'])?->first();
            if ($user) {
                $exist_in_convoy = $user->orcid_id ? ($this->getDiscloserByIdentifier($discloserListId, $user->orcid_id) ??  $this->getDiscloserByIdentifier($discloserListId,  $user->email)) : $this->getDiscloserByIdentifier($discloserListId, $user->email);
                if ($exist_in_convoy) {
                    $convey_user = $exist_in_convoy;
                } else {
                    if ($user->status) {
                        if ($user->status == 'a') {
                            $user->status = [
                                'id' => 'ACTIVE',
                                'description' => 'Active'
                            ];
                        } else {
                            $user->status = [
                                'id' => 'ARCHIVED',
                                'description' => 'Archived'
                            ];
                        }
                    }
                }
                // if($user->middleName){
                //     $user->firstName = $user->firstName + $user->middleName;
                //     unset($user->middleName);
                // }
                if ($role) {
                    $user->role = implode($role);
                }
                $convey_user = $this->createDiscloser($discloserListId, $user->toArray());
            }
            if ($convey_user) {
                $resp_data = [
                    'rcd_usr_id' => $user_id,
                    'rcd_art_id' => $art_id,
                    'rcd_discloser_id' => $convey_user['id'],
                    'rcd_discloser_list_id' => $discloserListId,
                ];
                ConveyData::create($resp_data);
            }
        } else {
            $this->updateUserStatusInDislcosureList($art_id, $user_id, 'active');
        }
        if (($existing_discloser && $existing_discloser->rcd_disclosure_cmpltn_status !== 'c') || $convey_user) {
            (new Manage_Email_Api())->sendConveyDisclosureEmail($existing_discloser->rcd_discloser_id ?? $convey_user['id'], 'send_convey_disclosure_link');
        }

        return;
    }

    public function createDiscloserListForArticle($art_id): array
    {
        $article = Articles::with('article_types')->find($art_id, ['art_id', 'art_artp_id', 'art_code']);
        $art_code = $article->art_code;
        $journal = Journals::find($article->article_types->artp_jnl_id, ['jnl_id', 'jnl_journal_name']);
        $group = $this->createGroupsForJournal($journal->jnl_id, $journal->jnl_journal_name);
        $discloserListData = [
            'name' => $art_code,
            'vendor' => 'RVT Review System',
            'purpose' => "CoI for manuscript $art_code",
            'contactName' => 'Contact Name', //need change to dynamic
            'contactEmail' => 'examplemail@rivervalleytechnologies.com',
            'submissionDeadline' => Carbon::now()->addYear(), //need to be dynamic
            'status' => ['id' => 'ACTIVE', 'description' => 'Active'],
            'listGroupIds' => [$group['id']]
        ];
        $resp =  $this->createDiscloserList($discloserListData);
        return $resp;
    }

    public function updateArticleDiscloserList($art_id, $data)
    {
        $discloser_list_id = ConveyData::where('rcd_art_id', $art_id)->get(['rcd_discloser_list_id'])?->first()?->rcd_discloser_list_id;
        if ($discloser_list_id) {
            $discloser_list_data = $this->getDiscloserListDetail($discloser_list_id);
            foreach ($data as $key => $value) {
                $discloser_list_data[$key] = $value;
            }

            return $this->updateDiscloserList($discloser_list_id, $discloser_list_data);
        }
    }

    public function createGroupsForJournal($jnl_id, $jnl_name = null, $throw_error_if_exist = false)
    {
        $resp = $this->listGroups(['vendorId' => "RVT_Journal-$jnl_id"]);
        $existingGroup = $resp['content'];
        if (count($existingGroup)) {
            if ($throw_error_if_exist) {
                throw new Exception('Group for this journal ready exists.');
            }
            $group = $existingGroup[0];
        } else {
            if (!$jnl_name) {
                $jnl_name = Journals::find($jnl_id, ['jnl_journal_name'])->jnl_journal_name;
            }
            $group = $this->createGroups([
                'name' => $jnl_name,
                'vendorId' => "RVT_Journal-$jnl_id",
            ]);
        }
        return $group;
    }

    public function getDiscloserData($user_id, $art_id)
    {
        $discloser_data = ConveyData::where(['rcd_usr_id' => $user_id, 'rcd_art_id' => $art_id])->get(['rcd_discloser_list_id'])->first();
        $resp_data = null;
        if ($discloser_data) {
            $identifier = User::find($user_id)?->usr_email;
            if ($identifier) {
                $resp_data = $this->getDiscloserByIdentifier($discloser_data->rcd_discloser_list_id, $identifier);
            }
        }
        if ($resp_data) {
            $resp_data = $this->formatContent($resp_data);
        }
        return $resp_data;
    }

    private function formatContent($data)
    {
        if (is_array($data)) {
            foreach ($data as $key => $val) {
                if (str_contains($key, 'subscriber')) {
                    if ($key == 'subscriberId') {
                        $data['identifier'] = $val;
                    } else {
                        $data[Str::camel((str_replace('subscriber', '', $key)))] = $val;
                    }
                    unset($data[$key]);
                }
            }

            if ($data['disclosureSubmitted'] || $data['inProgress']) {
                $data['disclosureStatus'] = $data['inProgress'] ? ['id' => 'inProgress', 'description' => 'In Progress'] : ['id' => 'completed', 'description' => 'Completed'];
            } else {
                $data['disclosureStatus'] = ['id' => 'invited', 'description' => 'Invited'];
            }

            $this->updateDisclosureStatusInLocal($data['id'], $data['disclosureStatus']);

        }
        return $data;
    }

    public function disclosersInArticle($art_id)
    {
        $data = ConveyData::where('rcd_art_id', $art_id)->get(['rcd_discloser_list_id'])?->first();
        $disclosers = [];
        if ($data) {
            $disclosers = $this->getDisclosersFromList($data->rcd_discloser_list_id);
        }
        if ($disclosers) {
            foreach ($disclosers as $key => $discloser) {
                if ((($discloser['status']['id'] ?? null) == 'ACTIVE' || ($discloser['active'] && !$discloser['archived']))) {
                    $disclosers[$key] = $this->formatContent($discloser);
                }else{
                    unset($disclosers[$key]);
                }
            }
        }
        return $disclosers;
    }

    public function addAllUsersOfArticle($art_id)
    {
        $authors = Authors::where('auth_art_id', $art_id)->get(['auth_usr_id', 'auth_corr_author']);
        $submitted_author = Articles::find($art_id, ['art_submitted_by'])?->art_submitted_by;

        foreach ($authors as $author) {
            $author_type = [];
            if ($author->auth_corr_author == 'y') {
                $author_type = ['Corresponding author'];
            } elseif ($author->auth_usr_id == $submitted_author) {
                $author_type = ['Submitting author'];
            } else {
                $author_type = ['Co-author'];
            }
                $this->addUserToDiscloser($author->auth_usr_id, $art_id, $author_type);
        }

        return;
    }

    private function conveyXMLParser($xmlContent)
    {
        $service = new XMLService();

        $encoding = mb_detect_encoding($xmlContent, ['UTF-8', 'ISO-8859-1', 'Windows-1252'], true);
        $xmlContent = mb_convert_encoding($xmlContent, 'UTF-8', $encoding);

        // Mapping namespaces for better readability
        $service->namespaceMap = [
            'http://ns.medbiq.org/Disclosure/v1/' => 'disclosure',
            'http://www.w3.org/2004/08/xop/include' => 'xop',
            'http://www.w3.org/1999/xlink' => 'xlink',
            'http://convey.aamc.org/conveydisclosure/v2/' => 'convey',
            'http://ns.medbiq.org/common/v2/' => 'common',
            'http://ns.medbiq.org/name/v1/' => 'name',
            'http://ns.medbiq.org/address/v1/' => 'address',
            'http://ns.medbiq.org/member/v1/' => 'member',
        ];

        $parsedData = $service->parse($xmlContent);

        // Post-process the parsed data
        return $this->cleanData($parsedData);
    }

    private function cleanData(array $data)
    {
        $result = [];

        foreach ($data as $item) {
            $key = $this->stripNamespace($item['name']);
            $value = is_array($item['value']) ? $this->cleanData($item['value']) : $this->stripHtml($item['value']);

            if (!empty($item['attributes'])) {
                $value = ['value' => $value, 'attributes' => $item['attributes']];
            }

            $result[$key][] = $value; // Use an array to handle multiple occurrences of the same key
        }

        // Simplify if only one occurrence of a key
        foreach ($result as $key => $values) {
            if (count($values) === 1) {
                $result[$key] = $values[0];
            }
        }

        return $result;
    }

    private function stripNamespace($name)
    {
        // Remove namespace prefixes or braces
        if (strpos($name, '}') !== false) {
            return substr($name, strpos($name, '}') + 1);
        }

        return $name;
    }

    private function stripHtml($value)
    {
        // Strip HTML tags if the value is a string
        return is_string($value) ? preg_replace('/<br\s*\/?>$/i', '', strip_tags($value, '<br>')) : $value;
    }

    function transformAdditionalQuestions(array $data): array
    {
        $result = [];

        // Helper function to extract child questions
        $extractChildQuestions = function ($childQuestions) {
            $childResult = [];
            foreach ($childQuestions as $type => $questions) {
                // foreach ($questions as $question) {
                $childResult[] = [
                    'QuestionText' => $questions['QuestionText'],
                    'Answer' => $questions['Answer'],
                    'displayOrder' => (int) $questions['DisplayOrder'],
                ];
                // }
            }

            usort($childResult, function ($a, $b) {
                return $a['displayOrder'] <=> $b['displayOrder'];
            });

            return $childResult;
        };

        // Process AdditionalQuestions
        foreach (['ShortAnswer', 'Conditional'] as $section) {
            if (isset($data[$section])) {
                foreach ($data[$section] as $question) {
                    $entry = [
                        'QuestionText' => $question['QuestionText'],
                        'Answer' => $question['Answer'],
                        'displayOrder' => (int) $question['DisplayOrder'],
                    ];

                    if (isset($question['ChildQuestions'])) {
                        $entry['ChildQuestions'] = $extractChildQuestions($question['ChildQuestions']);
                    }

                    $result[] = $entry;
                }
            }
        }

        // Process DataDrivenQuestion
        if (isset($data['DataDrivenQuestion'])) {
            $ddq = $data['DataDrivenQuestion'];
            $entry = [
                'QuestionText' => $ddq['QuestionText'],
                'Answer' => [],
                'displayOrder' => (int) $ddq['DisplayOrder'],
            ];

            foreach ($ddq['DataDrivenQuestion'] as $item) {
                $entityName = $item['attributes']['EntityName'] ?? '';
                $childQuestions = $item['value']['Section']['ChildQuestions'] ?? [];

                $entry['Answer'][] = [
                    'EntityName' => $entityName,
                    'ChildQuestions' => $extractChildQuestions($childQuestions),
                ];
            }

            $result[] = $entry;
        }

        usort($result, function ($a, $b) {
            return $a['displayOrder'] <=> $b['displayOrder'];
        });

        return $result;
    }


    public function FormatParsedXML($data)
    {
        $return_data = [];
        $return_data['purpose'] = $data['IndividualFinancialInterestPackage']['Metadata']['DisclosureProfile'];
        $return_data['name'] = $data['IndividualFinancialInterestPackage']['DisclosingIndividual']['Name']['FormattedName'];
        $return_data['email'] = $data['IndividualFinancialInterestPackage']['DisclosingIndividual']['Email'];
        $return_data['previous_disclosure'] = $data['IndividualFinancialInterestPackage']['DisclosureHistory']['PreviousDisclosureId']['value'] ?? null;
        $questionsFormater = $this->transformAdditionalQuestions($data['IndividualFinancialInterestPackage']['AdditionalQuestions']);
        $return_data['addlnQstn'] = $questionsFormater;

        return $return_data;
    }

    public function retriveAdtnlQstns($disclosureId)
    {
        return $this->getDisclosureObject($disclosureId);
    }

    public function getDownLoadLink($discloserId, $disclosureId)
    {
        $discloser_data = ConveyData::where('rcd_discloser_id', $discloserId)->get(['rcd_discloser_list_id'])?->first();
        if ($discloser_data) {
        $data = $this->getFinalDisclosurePDFWithDocs($discloser_data->rcd_discloser_list_id, $discloserId, $disclosureId);

        return $data['location'];
        }
    }

    public function updateUserStatusInDislcosureList($art_id, $user_id, $status)
    {
        $discloser = ConveyData::where(['rcd_art_id' => $art_id, 'rcd_usr_id' => $user_id])->get(['rcd_discloser_id', 'rcd_discloser_list_id'])->first();
        if ($discloser) {
            $discloserDetails = $this->getDiscloserDetail($discloser->rcd_discloser_list_id, $discloser->rcd_discloser_id);
            if ($discloserDetails) {
                if (strtolower($discloserDetails['status']['id']) != $status) {
                    unset($discloserDetails['id'],$discloserDetails['requestSent'], $discloserDetails['disclosureSubmitted'], $discloserDetails['disclosureId']);
                    if ($status == 'active') {
                        $discloserDetails['status'] = [
                            'id' => 'ACTIVE',
                            'description' => 'Active'
                        ];
                    } elseif ($status == 'archive') {
                        $discloserDetails['status'] = [
                            'id' => 'ARCHIVED',
                            'description' => 'Archived'
                        ];
                    }

                    $this->updateDiscloserDetail($discloser->rcd_discloser_list_id, $discloser->rcd_discloser_id, $discloserDetails);
                }
            }
        }
    }

    public function updateDisclosureStatusInLocal($discloser_id, $disclosure_status){
        $status = 'i';
        if($disclosure_status['id'] == 'inProgress'){
            $status = 'p';
        }elseif($disclosure_status['id'] == 'completed'){
            $status = 'c';
        }
        $update = ConveyData::where('rcd_discloser_id', $discloser_id)->update(['rcd_disclosure_cmpltn_status' => $status]);

        return $update;
    }

    public function getAllDisclosureList($filters){
        $params['includeArchived'] = ($filters['includeArchived'] ?? false ? $filters['includeArchived'] : "false");
        $params['sd'] = $filters['start_date'];
        $params['ed'] = $filters['end_date'];
        $params['page'] = ($filters['page'] ?? 0);
        $params['size'] = ($filters['size'] ?? 1);
        $params['sort'] = ($filters['sort'] ?? 'certifiedDate,desc');           //available sort : identifier,personalInfo.lastName,personalInfo.firstName,purpose,certifiedDate
        $data = $this->getAllDisclosureListApi($params);

        return $data;
    }
}
