<?php

namespace App\Http\Services;
use Illuminate\Support\Facades\Http;
use Exception;
use FontLib\Table\Type\nameRecord;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
class SuperSudoService
{
    private $basesupersudourl;
    private $authToken;

    public function __construct()
    {
        $this->basesupersudourl = config('superSudoServiceConfig.super_sudo_base_url')['production'] ?? '';
        $this->authToken = Cache::get('supersudo_auth_token', function () {
            
        return $this->authenticate();
        });
    }
    


        /**
     * Authenticate and retrieve token
     *
     * @return string
     */
    private function authenticate(): string
    {
        try
        {
        $username = config('superSudoServiceConfig.super_sudo_username') ?? null;
        $password = config('superSudoServiceConfig.super_sudo_password') ?? null;
        $auth_url = config('superSudoServiceConfig.super_sudo_authentication_url')['production'] ?? null;

        if (!$auth_url || !$username || !$password) {
            throw new Exception('Super sudo  service is not configured in the system, please check the super sudo service configurations.');
        }
        $response = Http::withHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ])
        ->asForm()
        ->post($auth_url, [
            'username' => $username,
            'password' => $password,
        ]);

        if ($response->successful()) {
            $newToken = $response->json('data.access_token');
            
            Cache::put('supersudo_auth_token', $newToken);
            return $newToken;
        }
        throw new \Exception('Authentication failed: ' . $response->body());
    }
    catch (Exception $e) {
        print_r($e->getMessage());
    }
    }


    public function get(string $endpoint, array $payload = [], $type = null, $use_web_url = false)
    {

    
            $response = Http::withHeaders([
                // 'Cache-Control' => 'no-cache',
                'Authorization' => 'Bearer ' . $this->authToken,
            ])->get("{$basesupersudourl}{$endpoint}", $payload);
            if ($response->status() === 401) {
                $this->authToken = $this->authenticate();
                throw new \Exception('Unauthorized (401) - Retrying...');
            }
            return $response;
       

        if ($response->successful()) {
            return $type ? $response->body() : $response->json();
        }

        throw new \Exception('GET request failed: ' . $response->body());
    }

    /**
     * Send POST request to the Convey API
     *
     * @param string $endpoint
     * @param array $payload
     * @return array
     * @throws \Exception
     */
    public function post(string $endpoint, array $payload): array
    {
        $response = retry(3, function () use ($endpoint, $payload) {
            $response = Http::withHeaders([
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->authToken,
            ])->post("{$this->basesupersudourl}{$endpoint}", $payload);

            if ($response->status() === 401) {
                $this->authToken = $this->authenticate();
                throw new \Exception('Unauthorized (401) - Retrying...');
            }

            return $response;
        }, 100);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('POST request failed: ' . $response->body());
    }

       /**
     * Create client based functionality and status stored in vector db
     *
     
     * @param array $payload name, vendorId(optional)
     * @return array
     */
    public function update_client_functionality_vector_db(array $payload = []): array
    {
        /*$payload=[
            'funct_name' => 'cfty_flag_icon_selector',
            'funct_status' => "n",
        ];*/
        return $this->post("/store-client-functionality", $payload);

    }

    public function update_client_module_vector_db(array $payload = []): array
    {
        /*$payload=[
            'funct_name' => 'cfty_flag_icon_selector',
            'funct_status' => "n",
        ];*/
        
        return $this->post("/store-module-functionality", $payload);

    }

    public function encode_xml_rpc(array $payload = []): array
    {
        $response = Http::post($this->basesupersudourl . "/encode-xml-rpc", $payload);

        if ($response->successful()) {
            return $response->json(); // returns decoded JSON as an array
        }
    
        // Handle failure (optional)
        return [
            'error' => true,
            'status' => $response->status(),
            'message' => $response->body()
        ];
    }
    public function decode_xml_rpc($payload )
    {

        $response = Http::withHeaders([
            'Content-Type' => 'text/xml'
        ])->send('POST', $this->basesupersudourl . "/decode-xml-rpc", [
            'body' => $payload, // must be raw string
        ]);
       
       
        if ($response->successful()) {
         
            return $response->json(); // returns decoded JSON as an array
        }
    
        // Handle failure (optional)
        return [
            'error' => true,
            'status' => $response->status(),
            'message' => $response->body()
        ];
    }
    
}
