<?php
namespace App\Http\Traits;

use Illuminate\Http\Request;
use App\Models\Articles;
use DOMDocument;
use DOMXPath;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Config;
use Illuminate\Support\Str;

trait JatsArtcileXmlTrait {

    public function createJatsArchiveXmlTags($art_id, $meca_path_fldr, $dtdPath) 
    {
        $select         =   'art_id, art_code, art_doi, art_abstract, art_artp_id, artp_abbr_article_type, art_title, art_status, art_epub_date, art_ppub_date, art_version_no, art_delete_status, artp_article_type, jnl_id, jnl_journal_code, jnl_journal_name, jnl_metadata_id, jwf_id, aflw_id, jwf_stage_type, jwf_stage_name, jwf_status_name, jwf_tat_accept,jwf_role_id, jwf_parent_user_roles';
        $art_details    =   Articles::get_article_details($select,"art_id={$art_id}");

        if(count($art_details) > 0) {
            $data           =   $art_details[0];
            $art_code       =   $data->art_code;
            $jnl_id         =   $data->jnl_id;
                
            // JATS-compliant XML file that uses the Journal Archiving and Interchange Tag
            $file_location  =   $meca_path_fldr . '/' . $art_code . '.xml';
            $destination    =   $file_location;

            $xml_string = '<?xml version="1.0" encoding="UTF-8"?>
                            <!DOCTYPE article PUBLIC "-//NLM//DTD JATS (Z39.96) Journal Archiving and Interchange DTD v1.4 20241031//EN" "'.$dtdPath.'">';

            // article-type: Type or kind of article (for example, “research”, “commentary”, “review”, “case”, or “calendar”).
            // Use article-type="research" for articles presenting original research.
            $xml_string .=  '<article 
                            article-type="research-article" 
                            dtd-version="1.4" 
                            xmlns:xlink="http://www.w3.org/1999/xlink" 
                            xml:lang="en">
                            </article>';
            
            file_put_contents($file_location, $xml_string);

            // Load your JATS XML document into a DOMDocument object
            $doc = new \DOMDocument();
            $doc->preserveWhiteSpace    = false;        // Prevent unnecessary white spaces
            $doc->formatOutput          = true;         // Enable formatting

            $doc->load($destination);

            // Create an XPath instance
            $xpath = new \DOMXpath($doc);
            // Use XPath query to find all <article> elements in the document
            $root_element = $xpath->query('//article');

            $root_node = $root_element->item(0);
            
            // Holds the metadata about the processing or technical handling of the document. 
            $processingArr          =   array('tagset-family'=>"jats", 'base-tagset'=>"archiving", 'table-model'=>"xhtml");
            $processing_meta_node   =   $this->createAndAppend($doc, $root_node, 'processing-meta', $processingArr);

            // if metadata tags are available, use custom meta tags
            $custom_meta_grp_node   =   $this->createAndAppend($doc, $processing_meta_node, 'custom-meta-group');
            $custom_meta_node       =   $this->createAndAppend($doc, $custom_meta_grp_node, 'custom-meta', ['xlink:href'=>'https://rivervalley.io/']);
            $this->createAndAppend($doc, $custom_meta_node, 'meta-name', [], 'Assigning authority');
            $this->createAndAppend($doc, $custom_meta_node, 'meta-value', [], 'River Valley Technologies'); 


            //Contains front matter elements such as journal metadata, article metadata, and abstract.
            $front = $this->createAndAppend($doc, $root_node, 'front'); 
            
            //defines the characteristics and identity of a journal such as the journal title, ISSN, publisher details etc
            $jnl_meta_node = $this->createAndAppend($doc, $front, 'journal-meta');
            $this->createAndAppend($doc, $jnl_meta_node, 'journal-id', [], $this->specialChars($data->jnl_journal_code));
            $jnl_title_grp_node = $this->createAndAppend($doc, $jnl_meta_node, 'journal-title-group');
            $this->createAndAppend($doc, $jnl_title_grp_node, 'journal-title', [], $this->specialChars($data->jnl_journal_name));

            //showing journal metadata
            $metadata_tags_arr =   array('issn', 'isbn', 'notes', 'issn-l');
            //there is only tags like <issn>, <issn-l>, <notes>, <isbn> to show metadata, of others we has to show under <custom-meta>
            if($data->jnl_metadata_id != NULL) {
                $gdata_tbl      =   'rvw_article_gnf_form_data_'.$data->jnl_journal_code;
                if (Schema::hasTable($gdata_tbl)) {
                    $Generic_forms = new \App\Models\Generic_forms();
                    $metadata_res    =   $Generic_forms::get_records_by_raw_query('SELECT gdata_label, gdata_data, gdata_display_data FROM ' . $gdata_tbl . ' WHERE gdata_atf_id='.$data->jnl_metadata_id);
                    if(count($metadata_res) > 0) {
                        // if metadata tags are available
                        foreach($metadata_res as $key => $arr) {
                            // Removes special chars from metadata labels
                            $gdata_label        =   preg_replace('/[^A-Za-z0-9]/', '', strtolower($arr->gdata_label)); 
                            if($arr->gdata_data != '' && in_array($gdata_label, $metadata_tags_arr)) {
                                $this->createAndAppend($doc, $jnl_meta_node, $gdata_label, [], $this->specialChars($arr->gdata_display_data));       
                            }
                        }
                    }
                }
            }

            //Container element for information about the person, company, or other entity that published a work.
            $publisher_node = $this->createAndAppend($doc, $jnl_meta_node, 'publisher'); 
            $this->createAndAppend($doc, $publisher_node, 'publisher-name', [], $this->specialChars($this->current_client->clnt_client_name)); 
            if($this->current_client->clnt_client_country != '') {
                $this->createAndAppend($doc, $publisher_node, 'publisher-loc', [], $this->specialChars($this->current_client->clnt_client_country)); 
            }

            //Container element for information concerning the article that identifies or describes the article.
            $art_meta_node = $this->createAndAppend($doc, $front, 'article-meta');
            // article code
            $this->createAndAppend($doc, $art_meta_node, 'article-id', ['custom-type'=>'manuscript'], $data->art_code);
            // article doi
            if($data->art_doi != '' || $data->art_doi != NULL)   {
                $this->createAndAppend($doc, $art_meta_node, 'article-id', ['pub-id-type'=>'doi'], $data->art_doi);
            }
            // Specify the nature of a particular version of an article.
            if($data->art_version_no > 0)
            {
                $art_version_no =   'Revision ' . $data->art_version_no + 1;
                $art_verArr     =   ['article-version-type'=>"revision-".$data->art_version_no + 1];
            }
            else
            {
                $art_version_no =   'Original version';
                $art_verArr     =   ['article-version-type'=>"original"];
            }
            $this->createAndAppend($doc, $art_meta_node, 'article-version', $art_verArr, $art_version_no);

            // container for elements that may be used to group articles such as article type, category, keywords, classifications
            $art_cat_node = $this->createAndAppend($doc, $art_meta_node, 'article-categories');
            // container for article type
            if($data->artp_article_type != '') {
                $artyp_grp_node = $this->createAndAppend($doc, $art_cat_node, 'subj-group', ['subj-group-type'=>'Article Type']);
                $this->createAndAppend($doc, $artyp_grp_node, 'subject', ['id' =>$this->specialChars($data->artp_abbr_article_type)], $this->specialChars($data->artp_article_type));
            }
            // container for classifications
            $Article_classification = new \App\Models\Article_classification();
            $classiArr  =   $Article_classification::get_tree_model_art_classi_arr($data->art_id);
            if(count($classiArr) > 0) {
                foreach($classiArr as $key => $val) {
                    // Parent Classification
                    $classi_grp_node = $this->createAndAppend($doc, $art_cat_node, 'subj-group', ['subj-group-type'=>'Classification']);
                    //avoid ungrouped keys from the array of classifications
                    $this->createAndAppend($doc, $classi_grp_node, 'subject', [], $this->specialChars($val->term_name));
                    if($val->child_cnt != null) {
                        $this->show_sub_classi($doc, $val->child_array, $classi_grp_node);
                    }
                }
            }

            //  Hold the main title of the article.
            $tit_grp_node = $this->createAndAppend($doc, $art_meta_node, 'title-group');
            $this->createAndAppend($doc, $tit_grp_node, 'article-title', [], $this->specialChars($data->art_title));
 
            // Container for one or more contributors and information about those contributors.
            $Authors = new \App\Models\Authors();
            $article_authors    =   $Authors::get_art_all_author_details($data->art_id);
            if(count($article_authors) > 0) {
                $contrib_grp_node   =   $this->createAndAppend($doc, $art_meta_node, 'contrib-group', array('content-type'=>"authors"));
                $affArr             =   array();
                foreach($article_authors as $key => $val) 
                {
                    // Represents a single contributor.
                    $contribAtrr    =   ['contrib-type'=>'author'];
                    if($val->auth_corr_author == 'y')
                        $contribAtrr['corresp']  =   'yes';
                    $contrib_node   =   $this->createAndAppend($doc, $contrib_grp_node, 'contrib', $contribAtrr, '');

                    //  Uniform Resource Identifier (URI)
                    if($val->usr_orcid_id != '' || $val->usr_orcid_id != NULL)
                    {
                        $this->createAndAppend($doc, $contrib_node, 'contrib-id', ['contrib-id-type'=>"orcid"], "https://orcid.org/".$this->specialChars($val->usr_orcid_id));   
                    }

                    $contrib_name_node   =   $this->createAndAppend($doc, $contrib_node, 'name');
                    // specify the family name or last name of a person
                    $this->createAndAppend($doc, $contrib_name_node, 'surname', [], $this->specialChars($val->usr_last_name));
                    // specify the given names, also known as first names or forenames, of a person.
                    $usr_gvn_nam    =   $this->specialChars($val->usr_first_name);
                    if($val->usr_middle_name != '' || $val->usr_middle_name != NULL) {
                        $usr_gvn_nam    =   $this->specialChars($val->usr_first_name).' '.$this->specialChars($val->usr_middle_name);
                    }
                    $this->createAndAppend($doc, $contrib_name_node, 'given-names', [], $usr_gvn_nam);
                    // prefix
                    $this->createAndAppend($doc, $contrib_name_node, 'prefix', [], $this->specialChars($val->usr_title));

                    // Role or Function Title of Contributor
                    if($val->auth_corr_author == 'y')
                        $contr_role     =   'Corresponding author';
                    else
                        $contr_role     =   'Co-author';
                    $this->createAndAppend($doc, $contrib_node, 'role', [], $contr_role);

                    // Reference to affiliation within the document
                    $affArr['aff'.$key]     =   $val;
                    $xrefAttr               =   ['ref-type'=>'aff', 'rid'=>'aff'.$key];
                    $this->createAndAppend($doc, $contrib_node, 'xref', $xrefAttr);
                   
                    // Email
                    if($val->usr_email != '' || $val->usr_email != NULL)
                    {
                        $this->createAndAppend($doc, $contrib_node, 'email', [], $this->specialChars($val->usr_email));   
                    }
                }

                foreach($affArr as $affKey => $affVal) 
                {
                    // Affiliation details
                    $contrib_aff_node      =   $this->createAndAppend($doc, $contrib_grp_node, 'aff', ['id'=>$affKey], '');
                    // Institution
                    if($affVal->auaff_company != '' || $affVal->auaff_company != NULL)
                    {
                        if($affVal->auaff_ringgold_id != '') {
                            $instId     =   $affVal->auaff_ringgold_id;
                            $instIdTyp  =   'Ringgold';
                        }
                        elseif($affVal->auaff_ror_id != '') {
                            $instId     =   $affVal->auaff_ror_id;
                            $instIdTyp  =   'Ror';
                        }
                        else {
                            $instId     =   '';
                            $instIdTyp  =   '';
                        }
                        
                        if($instId != '') {
                            $inst_wrp_node      =   $this->createAndAppend($doc, $contrib_aff_node, 'institution-wrap');
                            $this->createAndAppend($doc, $inst_wrp_node, 'institution-id', ['institution-id-type' => $instIdTyp], $this->specialChars($instId));
                            $this->createAndAppend($doc, $inst_wrp_node, 'institution', ['content-type' => $affVal->inst_type], $this->specialChars($affVal->auaff_company));
                        }
                        else{
                            $this->createAndAppend($doc, $contrib_aff_node, 'institution', ['content-type' => $affVal->inst_type], $this->specialChars($affVal->auaff_company));
                        }
                    }
                    // Department
                    if($affVal->auaff_department != '' || $affVal->auaff_department !=  NULL)
                    {
                        $this->createAndAppend($doc, $contrib_aff_node, 'institution', ['content-type'=>'dept'], $this->specialChars($affVal->auaff_department));
                    }
                    // Address 1
                    if($affVal->auaff_address_line1 != ''){
                        $this->createAndAppend($doc, $contrib_aff_node, 'addr-line', ['content-type'=>'addrline1'], $this->specialChars($affVal->auaff_address_line1));
                    }
                    // Address 2
                    if($affVal->auaff_address_line2 != ''){
                        $this->createAndAppend($doc, $contrib_aff_node, 'addr-line', ['content-type'=>'addrline2'], $this->specialChars($affVal->auaff_address_line2));
                    }
                    // State
                    if($affVal->auaff_state != '' || $affVal->auaff_state != NULL)
                    {
                        $this->createAndAppend($doc, $contrib_aff_node, 'state', [], $this->specialChars($affVal->auaff_state));
                    }
                    // Country
                    if($affVal->cnt_country != '' || $affVal->cnt_country != NULL)
                    {
                        $this->createAndAppend($doc, $contrib_aff_node, 'country', [], $this->specialChars($affVal->cnt_country));
                    }
                    // Postal code
                    if($affVal->auaff_postal_code != '' || $affVal->auaff_postal_code != NULL)
                    {
                        $this->createAndAppend($doc, $contrib_aff_node, 'postal-code', [], $this->specialChars($affVal->auaff_postal_code));
                    }
                    // Phone
                    if($affVal->auaff_phone != '' || $affVal->auaff_phone != NULL)
                    {
                        $this->createAndAppend($doc, $contrib_aff_node, 'phone', [], $this->specialChars($affVal->auaff_phone));
                    }
                }
            }
            // Container element for notes about the article’s contributors.
            
            //Showing article elctronic & print pub date
            if($data->art_epub_date == '' && $data->art_ppub_date == '') {
                // If Publication Date Not available
                $this->createAndAppend($doc, $art_meta_node, 'pub-date-not-available');
            } else {
                // If Publication Date is available
                if($data->art_epub_date != NULL ) {
                    $etimestamp     =   strtotime($data->art_epub_date);
                    $epubArr         =   ['date-type'=>'pub', 'iso-8601-date'=>$data->art_epub_date, 'publication-format'=>'electronic'];
                    $pub_date_node  =   $this->createAndAppend($doc, $art_meta_node, 'pub-date', $epubArr, '');
                    $this->createAndAppend($doc, $pub_date_node, 'day', [], date("d", $etimestamp));
                    $this->createAndAppend($doc, $pub_date_node, 'month', [], date("m", $etimestamp));
                    $this->createAndAppend($doc, $pub_date_node, 'year', [], date("Y", $etimestamp));
                } 
                if($data->art_ppub_date != NULL ) {
                    $ptimestamp     =   strtotime($data->art_ppub_date);
                    $ppubArr         =   ['date-type'=>'pub', 'iso-8601-date'=>$data->art_ppub_date, 'publication-format'=>'print'];
                    $pub_date_node  =   $this->createAndAppend($doc, $art_meta_node, 'pub-date', $ppubArr, '');
                    $this->createAndAppend($doc, $pub_date_node, 'day', [], date("d", $ptimestamp));
                    $this->createAndAppend($doc, $pub_date_node, 'month', [], date("m", $ptimestamp));
                    $this->createAndAppend($doc, $pub_date_node, 'year', [], date("Y", $ptimestamp));
                }
            }

            // Issue details
            $Issue_management = new \App\Models\Issue_management();
            $article_issue    =   $Issue_management::get_art_issue_details($data->art_id)->toArray();
            if(count($article_issue) > 0) {
                $this->createAndAppend($doc, $art_meta_node, 'volume', [], $this->specialChars($article_issue[0]->vol_title));
                $this->createAndAppend($doc, $art_meta_node, 'issue', [], $this->specialChars($article_issue[0]->iss_number));
                $this->createAndAppend($doc, $art_meta_node, 'fpage', [], $this->specialChars(str_replace("f-","",$article_issue[0]->front)));
                $this->createAndAppend($doc, $art_meta_node, 'lpage', [], $this->specialChars(str_replace("b-","",$article_issue[0]->back)));
            } else {
                $this->createAndAppend($doc, $art_meta_node, 'volume', [], '');
                $this->createAndAppend($doc, $art_meta_node, 'issue', [], '');
                $this->createAndAppend($doc, $art_meta_node, 'fpage', [], '');
                $this->createAndAppend($doc, $art_meta_node, 'lpage', [], '');
            }

            $article_attachment_files   =   $this->show_art_attachments($art_id, $jnl_id);
            if(count($article_attachment_files) > 0) {
                $stor_path = storage_path('/app/public/uploads');
                // Contains supporting material such as an appendix, acknowledgment, glossary, or bibliographic reference list.
                $i  =   1;
                foreach($article_attachment_files as $key) {
                    $mimetype           =   '';
                    $mime_subtype       =   '';
                    $stor_file_path     =   $stor_path.'/'.$key->atmnt_orginal_upload_path .'/'. $key->atmnt_file_name;

                    if (file_exists($stor_file_path)) {
                        $mime_content_type  =   mime_content_type($stor_file_path);
                        if($mime_content_type != NULL) {
                            $mctArr = explode( '/', $mime_content_type);
                            $mimetype       =   $mctArr[0];
                            $mime_subtype   =   $mctArr[1];
                        }
                        $supp_node = $this->createAndAppend($doc, $art_meta_node, 'supplementary-material', array('id' => 'source-data-'.$i, 'content-type'=>"sdata", 'mimetype'=>$mimetype, 'mime-subtype'=>$mime_subtype, 'xlink:href'=>$key->atmnt_file_name));
                    } else {
                        $supp_node = $this->createAndAppend($doc, $art_meta_node, 'supplementary-material', array('id' => 'source-data-'.$i, 'content-type'=>"sdata"));
                    }
                    $this->createAndAppend($doc, $supp_node, 'label', [], 'Supplementary File ' . $i);
                    $caption_node   =   $this->createAndAppend($doc, $supp_node, 'caption');
                    $this->createAndAppend($doc, $caption_node, 'title', [], $key->atmnt_file_name);
                    $atmnt_designated_to    =   '';
                    if (in_array("p", json_decode($key->atmnt_designated_to))) {
                        $atmnt_designated_to    =   'Production';
                    }
                    if (in_array("r", json_decode($key->atmnt_designated_to))) {
                        if($atmnt_designated_to != '')
                            $atmnt_designated_to    .=   ', Review';
                        else
                        $atmnt_designated_to    =   'Review';
                    }
                    $this->createAndAppend($doc, $caption_node, 'p', [], 'File intended for: ' . $atmnt_designated_to);
                    $this->createAndAppend($doc, $caption_node, 'p', [], 'Submission type: ' . $key->submission_type);
                    $atmnt_document_size    =   round($key->atmnt_document_size / 1024, 2) . ' MB';
                    $this->createAndAppend($doc, $caption_node, 'p', [], 'File size: ' . $atmnt_document_size);
                    $this->createAndAppend($doc, $caption_node, 'p', [], 'Uploaded date: ' . $key->atmnt_file_uploaded_date);
                    $i++;
                }
            }

            // History dates include received date(s), accepted date(s), reviewed date(s), revision date(s), and other dates that may be important to the publisher
            $histArr  =   Articles::get_article_history($data->art_id);
            if($histArr) {
                $history_node   =   $this->createAndAppend($doc, $art_meta_node, 'history');
                $histdata        =   $histArr[0];
                //article received date
                if($histdata->received_date != NULL) {
                    $rdtimestamp     =   strtotime($histdata->received_date);
                    $dtArr           =   ['date-type'=>'received', 'iso-8601-date'=>date('Y-m-d', $rdtimestamp)];
                    $recdt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $recdt_node, 'day', [], date("d", $rdtimestamp));
                    $this->createAndAppend($doc, $recdt_node, 'month', [], date("m", $rdtimestamp));
                    $this->createAndAppend($doc, $recdt_node, 'year', [], date("Y", $rdtimestamp));
                }
                //article withdrawn date
                if($histdata->withdrawn_date != NULL) {
                    $wdtimestamp     =   strtotime($histdata->withdrawn_date);
                    $dtArr           =   ['date-type'=>'withdrawn', 'iso-8601-date'=>date('Y-m-d', $wdtimestamp)];
                    $wdrdt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $wdrdt_node, 'day', [], date("d", $wdtimestamp));
                    $this->createAndAppend($doc, $wdrdt_node, 'month', [], date("m", $wdtimestamp));
                    $this->createAndAppend($doc, $wdrdt_node, 'year', [], date("Y", $wdtimestamp));
                }
                //article review started date
                if($histdata->review_started_date != NULL) {
                    $rv1timestamp     =   strtotime($histdata->review_started_date);
                    $dtArr           =   ['date-type'=>'review-started', 'iso-8601-date'=>date('Y-m-d', $rv1timestamp)];
                    $revstardt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $revstardt_node, 'day', [], date("d", $rv1timestamp));
                    $this->createAndAppend($doc, $revstardt_node, 'month', [], date("m", $rv1timestamp));
                    $this->createAndAppend($doc, $revstardt_node, 'year', [], date("Y", $rv1timestamp));
                }
                //article review ended date
                if($histdata->review_ended_date != NULL) {
                    $rv2timestamp     =   strtotime($histdata->review_ended_date);
                    $dtArr           =   ['date-type'=>'review-ended', 'iso-8601-date'=>date('Y-m-d', $rv2timestamp)];
                    $revstopdt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $revstopdt_node, 'day', [], date("d", $rv2timestamp));
                    $this->createAndAppend($doc, $revstopdt_node, 'month', [], date("m", $rv2timestamp));
                    $this->createAndAppend($doc, $revstopdt_node, 'year', [], date("Y", $rv2timestamp));
                }
                //article revision requested date
                if($histdata->revision_req_date != NULL) {
                    $rrqtimestamp     =   strtotime($histdata->revision_req_date);
                    $dtArr           =   ['date-type'=>'rev-request', 'iso-8601-date'=>date('Y-m-d', $rrqtimestamp)];
                    $rreqdt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $rreqdt_node, 'day', [], date("d", $rrqtimestamp));
                    $this->createAndAppend($doc, $rreqdt_node, 'month', [], date("m", $rrqtimestamp));
                    $this->createAndAppend($doc, $rreqdt_node, 'year', [], date("Y", $rrqtimestamp));
                }
                //article revision received date
                if($histdata->revision_recd_date != NULL) {
                    $rrctimestamp     =   strtotime($histdata->revision_recd_date);
                    $dtArr           =   ['date-type'=>'rev-recd', 'iso-8601-date'=>date('Y-m-d', $rrctimestamp)];
                    $rrecdt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $rrecdt_node, 'day', [], date("d", $rrctimestamp));
                    $this->createAndAppend($doc, $rrecdt_node, 'month', [], date("m", $rrctimestamp));
                    $this->createAndAppend($doc, $rrecdt_node, 'year', [], date("Y", $rrctimestamp));
                }
                //article accepted date
                if($histdata->accepted_date != NULL) {
                    $adtimestamp     =   strtotime($histdata->accepted_date);
                    $dtArr           =   ['date-type'=>'accepted', 'iso-8601-date'=>date('Y-m-d', $adtimestamp)];
                    $accdt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $accdt_node, 'day', [], date("d", $adtimestamp));
                    $this->createAndAppend($doc, $accdt_node, 'month', [], date("m", $adtimestamp));
                    $this->createAndAppend($doc, $accdt_node, 'year', [], date("Y", $adtimestamp));
                }
                //article published date
                if($histdata->published_date != NULL) {
                    $pdtimestamp     =   strtotime($histdata->published_date);
                    $dtArr           =   ['date-type'=>'published', 'iso-8601-date'=>date('Y-m-d', $pdtimestamp)];
                    $pubdt_node      =   $this->createAndAppend($doc, $history_node, 'date', $dtArr, '');
                    $this->createAndAppend($doc, $pubdt_node, 'day', [], date("d", $pdtimestamp));
                    $this->createAndAppend($doc, $pubdt_node, 'month', [], date("m", $pdtimestamp));
                    $this->createAndAppend($doc, $pubdt_node, 'year', [], date("Y", $pdtimestamp));
                }
            }

            // Container element for copyright and license information of the article
            $cnd                =   'crt_art_id = '.$art_id;
            $Copyright          =   new \App\Models\Copyright();
            $copyright_details  =   $Copyright::get_copyright($cnd, 'crtm_license_name, crtm_payment, crt_posted_date, crtm_license_details, copy_location, copy_att_name', TRUE, TRUE);
            if (!empty($copyright_details))
            {
                $copyright_yr   =   date("Y", strtotime($copyright_details->crt_posted_date));
                $perm_node   =   $this->createAndAppend($doc, $art_meta_node, 'permissions');
                $this->createAndAppend($doc, $perm_node, 'copyright-statement', [], 'Copyright License: ' . $copyright_details->crtm_license_name);
                $this->createAndAppend($doc, $perm_node, 'copyright-year', [], $copyright_yr);
                $this->createAndAppend($doc, $perm_node, 'copyright-holder', [], $copyright_details->usr_name);
                $copyright_mode_array=$this->Utility('copyright_mode_array');
                $licence_mode   =   $copyright_mode_array['copyright_mode_array'][$copyright_details->crtm_payment];
                $lics_node   =   $this->createAndAppend($doc, $perm_node, 'license', ['license-type'=>$licence_mode]);
                $this->createAndAppend($doc, $lics_node, 'license-p', [], $copyright_details->crtm_license_details);
            }

            // Summarized description of the article.
            $artAbstract = str_replace(['<p>', '</p>'], '', $data->art_abstract);
            $abstract_node  =   $this->createAndAppend($doc, $art_meta_node, 'abstract');
            $this->createAndAppend($doc, $abstract_node, 'p', [], $artAbstract);

            // container for keywords
            $Article_keywords = new \App\Models\Article_keywords();
            $articleKeywords = $Article_keywords::get_article_keywords($data->art_id);
            // Group keywords by 'kwd_user_keyword'
            $grouped = $articleKeywords->groupBy('kwd_user_keyword');
            // Convert to arrays for easier handling
            $groupedKwdArray = [
                'n' => $grouped->get('n', collect())->toArray(),
                'y' => $grouped->get('y', collect())->toArray(),
            ];
            
            // Output grouped Keywords array
            if(count($groupedKwdArray) > 0)
            {
                if(count($groupedKwdArray['n']) > 0)
                {
                    $kwd_grp_node = $this->createAndAppend($doc, $art_meta_node, 'kwd-group', ['kwd-group-type'=>'Controlled keywords']);
                    foreach($groupedKwdArray['n'] as $kkey => $kval) {
                        $this->createAndAppend($doc, $kwd_grp_node, 'kwd', [], $this->specialChars($kval->kwd_name));
                    }
                }

                if(count($groupedKwdArray['y']) > 0)
                {
                    $kwd_grp_node = $this->createAndAppend($doc, $art_meta_node, 'kwd-group', ['kwd-group-type'=>'Custom keywords']);
                    foreach($groupedKwdArray['y'] as $kkey => $kval) {
                        $this->createAndAppend($doc, $kwd_grp_node, 'kwd', [], $this->specialChars($kval->kwd_name));
                    }
                }
            }

            // Contains the funding sources that supported the research
            $Article_funder_details = new \App\Models\Article_funder_details();
            $select_fund = 'funder_id, afd_grant_number, funder_unique_id, funder_name, funder_uri, funder_location';
            $article_funding_data = $Article_funder_details::get_article_funder_details($select_fund, array(
                'afd_art_id' => $art_id
            ));
            if(count($article_funding_data) > 0) {
                $fundgrp_code   =   $this->createAndAppend($doc, $art_meta_node, 'funding-group', ['specific-use' =>"crossref"]);
                $funders        =   '';
                $iter           =   1;
                foreach($article_funding_data as $key => $val) {
                    $awrdgrp_node   =   $this->createAndAppend($doc, $fundgrp_code, 'award-group', ['id'=>'award'.($iter)]);
                    $fundsrcArr     =   array('id'=>'funder'.$val->funder_id);
                    if($val->funder_location != '' || $val->funder_location != NULL)
                        $fundsrcArr['country'] = $val->funder_location;
                    $this->createAndAppend($doc, $awrdgrp_node, 'funding-source', $fundsrcArr, $val->funder_name);
                    if($val->afd_grant_number != '')
                        $this->createAndAppend($doc, $awrdgrp_node, 'award-id', [], $val->afd_grant_number);
                    // for funding stmt
                    $funders    .=   $val->funder_name .' ';
                    if($val->afd_grant_number != '')
                        $funders    .=   'under Grant No. ' . $val->afd_grant_number; 

                    if(count($article_funding_data) > 1) {
                        if($iter < (count($article_funding_data) -1))
                            $funders    .=   ', ';
                        else
                            $funders    .=   ', and ';
                    }
                    $iter++;
                }
                $funders = rtrim($funders, ', and ');
                // statement that describes the funding for the research on which a work was based
                $this->createAndAppend($doc, $fundgrp_code, 'funding-statement', [], 'This work was supported by '.$funders);
            }

            // version type, version status, version statement, or version number for the current article.
            $version_typ  =   $this->get_the_article_status($data->art_status, $data->art_delete_status, $this->current_client);
            // Indicates the type of version. Common values include "submitted", "revised", "accepted", "published".
            $custom_meta_grp_node  =   $this->createAndAppend($doc, $art_meta_node, 'custom-meta-group');
            if($version_typ != '') {
                $custom_meta_node  =   $this->createAndAppend($doc, $custom_meta_grp_node, 'custom-meta');
                $this->createAndAppend($doc, $custom_meta_node, 'meta-name', [], 'Article Status');
                $this->createAndAppend($doc, $custom_meta_node, 'meta-value', [], $version_typ); 
            }

            // if metadata tags are available, use custom meta tags
            $custom_meta_grp_node  =   $this->createAndAppend($doc, $art_meta_node, 'custom-meta-group');
            foreach($metadata_res as $key => $arr) {
                // Removes special chars from metadata labels
                $gdata_label        =   preg_replace('/[^A-Za-z0-9]/', '', strtolower($arr->gdata_label)); 
                if($arr->gdata_data != '' && !in_array($gdata_label, $metadata_tags_arr)) {
                    $custom_meta_node  =   $this->createAndAppend($doc, $custom_meta_grp_node, 'custom-meta');
                    $this->createAndAppend($doc, $custom_meta_node, 'meta-name', [], $this->specialChars($arr->gdata_label));
                    $this->createAndAppend($doc, $custom_meta_node, 'meta-value', [], $this->specialChars($arr->gdata_display_data)); 
                }
            }

            // save the xml file
            $doc->saveXML();
            // Save the XML document to the specified file location
            $doc->save($file_location);

            return array('xml'=>$file_location);
        } else {
            return array();
        }
    }


    public function createJatsManifestXmlTags($tags_arr, $meca_path_fldr, $dtdPath)
    {
        // JATS-compliant XML file that uses the Journal Archiving and Interchange Tag
        $file_location  =   $meca_path_fldr . '/MECA_manifest.xml';
        $destination    =   $file_location;

        $xml_string = '<?xml version="1.0" encoding="UTF-8"?>
                        <!DOCTYPE manifest PUBLIC "-//MECA//DTD Manifest v1.0//en" "'.$dtdPath.'">';

        // article-type: Type or kind of article (for example, “research”, “commentary”, “review”, “case”, or “calendar”).
        // Use article-type="research" for articles presenting original research.
        $xml_string .=  '<manifest 
                        manifest-version="1" 
                        xmlns:xlink="http://www.w3.org/1999/xlink"></manifest>';
        
        file_put_contents($file_location, $xml_string);

        // Load your JATS XML document into a DOMDocument object
        $doc = new \DOMDocument();
        $doc->preserveWhiteSpace    = false;        // Prevent unnecessary white spaces
        $doc->formatOutput          = true;         // Enable formatting

        $doc->load($destination);

        // Create an XPath instance
        $xpath = new \DOMXpath($doc);
        // Use XPath query to find all <manifest> elements in the document
        $root_element = $xpath->query('//manifest');

        $root_node = $root_element->item(0);

        $i  =   1;
        foreach($tags_arr as $key => $val)
        {
            $item_type      =   '';
            $item_descr     =   '';
            $inst_media_typ     =   '';
            $inst_xlink_href     =   '';
            if(isset($val['item_type']) && $val['item_type'] != '') {
                $item_type  =   $val['item_type'];
            }
            $itemArr            =   array('id'=>$i, 'item-type'=>$item_type);
            $item_meta_node     =   $this->createAndAppend($doc, $root_node, 'item', $itemArr);
            if(isset($val['item_descr']) && $val['item_descr'] != '') {
                $item_descr  =   $val['item_descr'];
            }
            $this->createAndAppend($doc, $item_meta_node, 'item-description', [], $item_descr);
            if(isset($val['inst_media_typ']) && $val['inst_media_typ'] != '') {
                $inst_media_typ  =   $val['inst_media_typ'];
            }
            if(isset($val['inst_xlink_href']) && $val['inst_xlink_href'] != '') {
                $inst_xlink_href  =   $val['inst_xlink_href'];
            }
            $instArr            =   array('media-type'=> $inst_media_typ, 'xlink:href'=>$inst_xlink_href);
            $this->createAndAppend($doc, $item_meta_node, 'instance', $instArr);
            $i++;
        }

        // save the xml file
        $doc->saveXML();
        // Save the XML document to the specified file location
        $doc->save($file_location);

        return array('xml'=>$file_location);
    }

    /*
    * Helper function to create and append an element
    * $doc - Main doc
    * $parent - parent tag
    * $tag - child (current) tag
    * $attributes - attributes that need to show inside the current tag
    * $text - value of the tag
    */
    private function createAndAppend($doc, $parent, $tag, $attributes = [], $text = '') {
        // creates an element
        $element = $doc->createElement($tag, htmlspecialchars($text));
        // set attributes to that element if any
        foreach ($attributes as $key => $value) {
            $element->setAttribute($key, $value);
        }
        // append the element to its parent element
        $parent->appendChild($element);
        return $element;
    }

    /*
    * show the sub classifications tags inside its parent tag
    */
    private function show_sub_classi($doc, $children =array(), $node) {
        $sub_grp_node   =   $this->createAndAppend($doc, $node, 'subj-group', ['subj-group-type'=>'Subclassification']);
        foreach($children as $key => $val) {
            $this->createAndAppend($doc, $sub_grp_node, 'subject', [], $this->specialChars($val->term_name));
            if(isset($val->children) && count($val->children) > 0) {
                $this->show_sub_classi($doc, $val->children, $sub_grp_node);
            }
        }
    }

    /*
    * fetch the article current status
    */
    private function get_the_article_status($art_status, $art_delete_status, $current_client) {
        $articleStatusQry   =   $this->getPermittedArticleStatus($current_client);
        $articleStatus      =   $articleStatusQry['articleStatus'];
        $article_status     =   ($art_delete_status == 'h') ? "On Hold" : (($art_delete_status == 'w') ? "Withdrawn" : (isset($articleStatus[$art_status]) ? $articleStatus[$art_status] : ''));
        return $article_status == '' ? $articleStatus['y'] : $article_status;
    }

    /*
    * ensure that all special characters in your strings are properly encoded
    */
    private function specialChars($text) {
        return htmlspecialchars($text, ENT_XML1, 'UTF-8');
    }

    /*
    * fetch the author submitted article attachments
    */
    private function show_art_attachments($art_id, $jnl_id) 
    {
        if(isset($jnl_id) && !empty($jnl_id)){
            $Attachment_files           =   new \App\Models\Attachment_files();
            $get_platform_date_format   =   $this->get_platform_date_format($jnl_id);
            $date_format_map            =   date_format_map($get_platform_date_format);
            $date_format                =   $date_format_map.' %H:%i:%s';

            $orgArt_path    =   '';
            $query_str      =   "SELECT a1.art_article_upload_path 
                                FROM `rvw_articles` as a1 
                                JOIN rvw_articles as a2 on (a2.`art_id`=? AND a1.art_doi = a2.art_doi) 
                                ORDER by a1.art_id LIMIT 1";
            $bind_arr       =   array($art_id);
            $result_orgpath =   Articles::get_records_by_raw_query($query_str, FALSE, $bind_arr);
            if(isset($result_orgpath[0])) {
                $orgArt_path = $result_orgpath[0]->art_article_upload_path;
            }

            $select     =   'atmnt_id,atmnt_art_id,atmnt_orginal_upload_path,atmnt_file_name,atmnt_document_size,atmnt_designated_to,atmnt_order,DATE_FORMAT(atmnt_file_uploaded_date, "' . $date_format . '") as atmnt_file_uploaded_date,IF(atmnt_orginal_upload_path = \''.$orgArt_path.'\',\'Original file\',\'Revised file\') as submission_type';
            $cond       =   array(
                                'atmnt_art_id' => $art_id
                            );

            return $Attachment_files::get_all_records_with_bind_cond($select, $cond,'',array(), 'atmnt_order');
        }
    }

    //Task #7443 Starts
    public static function jatsExportFunc($jatsClientEnable,$journal_setting,$role_id){
        if($jatsClientEnable == 'y'){
            $who_can_do = (isset($journal_setting->jset_jats_export) && $journal_setting->jset_jats_export!="")?json_decode($journal_setting->jset_jats_export):[];
            if(in_array($role_id,$who_can_do)){
                return true;
            }
        }
        return false;
    }
    //Task #7443 Ends
}