<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Get dynamic DB name from .env
        $dbName = env('DB_DATABASE');

        // Safely rename tables if they exist
        if (Schema::hasTable('rvw_unsuscribe_email')) {
            DB::statement("RENAME TABLE `{$dbName}`.`rvw_unsuscribe_email` TO `{$dbName}`.`rvw_unsubscribe_email`");
        }

        if (Schema::hasTable('tbk_unsuscribe_email')) {
            DB::statement("RENAME TABLE `{$dbName}`.`tbk_unsuscribe_email` TO `{$dbName}`.`tbk_unsubscribe_email`");
        }

        // Drop old unique index if exists (avoid error)
        Schema::table('tbk_unsubscribe_email', function (Blueprint $table) {
            try {
                $table->dropUnique('tbk_unsuscribe_email_usl_email_unique');
            } catch (\Exception $e) {
                // Ignore if index doesn't exist
            }
        });

        // Add new columns if missing
        if (Schema::hasTable('rvw_unsubscribe_email')) {
            Schema::table('rvw_unsubscribe_email', function (Blueprint $table) {
                if (!Schema::hasColumn('rvw_unsubscribe_email', 'usl_token')) {
                    $table->string('usl_token', 50)->nullable()->after('usl_comments');
                }
                if (!Schema::hasColumn('rvw_unsubscribe_email', 'usl_ipaddress')) {
                    $table->string('usl_ipaddress', 20)->after('usl_token');
                }
            });
        }

        if (Schema::hasTable('tbk_unsubscribe_email')) {
            Schema::table('tbk_unsubscribe_email', function (Blueprint $table) {
                if (!Schema::hasColumn('tbk_unsubscribe_email', 'usl_token')) {
                    $table->string('usl_token', 50)->nullable()->after('usl_comments');
                }
                if (!Schema::hasColumn('tbk_unsubscribe_email', 'usl_ipaddress')) {
                    $table->string('usl_ipaddress', 20)->after('usl_token');
                }
            });
        }
         // --- Drop existing triggers if they exist ---
        DB::unprepared('DROP TRIGGER IF EXISTS `trg_unsuscribe_email_after_insert`');
        DB::unprepared('DROP TRIGGER IF EXISTS `trg_unsuscribe_email_after_update`');

        // --- Create AFTER INSERT trigger ---
        DB::unprepared("
            CREATE TRIGGER `trg_unsuscribe_email_after_insert`
            AFTER INSERT ON `rvw_unsubscribe_email`
            FOR EACH ROW
            BEGIN
                INSERT INTO `tbk_unsubscribe_email`
                SELECT *, NULL AS tbk_id
                FROM `rvw_unsubscribe_email`
                WHERE `usl_id` = NEW.`usl_id`;
            END
        ");

        // --- Create AFTER UPDATE trigger ---
        DB::unprepared("
            CREATE TRIGGER `trg_unsuscribe_email_after_update`
            AFTER UPDATE ON `rvw_unsubscribe_email`
            FOR EACH ROW
            BEGIN
                IF NEW.`usl_operation` != 'm' THEN
                    INSERT INTO `tbk_unsubscribe_email`
                    SELECT *, NULL AS tbk_id
                    FROM `rvw_unsubscribe_email`
                    WHERE `usl_id` = OLD.`usl_id`;
                END IF;
            END
        ");

    }

    public function down(): void
    {
        $dbName = env('DB_DATABASE');

        // Reverse rename
        if (Schema::hasTable('rvw_unsubscribe_email')) {
            DB::statement("RENAME TABLE `{$dbName}`.`rvw_unsubscribe_email` TO `{$dbName}`.`rvw_unsuscribe_email`");
        }

        if (Schema::hasTable('tbk_unsubscribe_email')) {
            DB::statement("RENAME TABLE `{$dbName}`.`tbk_unsubscribe_email` TO `{$dbName}`.`tbk_unsuscribe_email`");
        }

        // Drop added columns if needed
        foreach (['rvw_unsuscribe_email', 'tbk_unsuscribe_email'] as $tableName) {
            if (Schema::hasTable($tableName)) {
                Schema::table($tableName, function (Blueprint $table) use ($tableName) {
                    if (Schema::hasColumn($tableName, 'usl_token')) {
                        $table->dropColumn('usl_token');
                    }
                    if (Schema::hasColumn($tableName, 'usl_ipaddress')) {
                        $table->dropColumn('usl_ipaddress');
                    }
                });
            }
        }
    }
};
